<?php
/**
 * External Sources Dribbble Class
 * @package   Essential_Grid
 * @author    ThemePunch <info@themepunch.com>
 * @link      http://www.themepunch.com/essential/
 * @copyright 2021 ThemePunch
 * @since: 3.0.13
 **/

if(!defined('ABSPATH')) exit();

/**
 * Dribbble
 *
 * with help of the API this class delivers all kind of Images/Projects from Dribbble
 *
 * @package    socialstreams
 * @subpackage socialstreams/dribbble
 * @author     ThemePunch <info@themepunch.com>
 */
class Essential_Grid_Dribbble
{
	/**
	 * Stream Array
	 *
	 * @since    3.0
	 * @access   private
	 * @var      array $stream Stream Data Array
	 */
	private $stream;

	/**
	 * Client Access Token
	 *
	 * @since    3.0
	 * @access   private
	 * @var      string $client_access_token dribbble API Client Access Token
	 */
	private $client_access_token;

	/**
	 * User ID
	 *
	 * @since    3.0
	 * @access   private
	 * @var      string $user_id dribbble User ID
	 */
	private $user_id;

	/**
	 * Transient seconds
	 *
	 * @since    3.0
	 * @access   private
	 * @var      number $transient Transient time in seconds
	 */
	private $transient_sec;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    3.0
	 */
	public function __construct($client_access_token, $user_id, $transient_sec = 0)
	{
		$this->user_id = $user_id;
		$this->client_access_token = $client_access_token;
		$this->transient_sec = 0;
		$this->stream = array();
	}

	/**
	 * Get Behance User Projects
	 *
	 * @since    3.0
	 */
	public function get_dribbble_projects($count = 100)
	{
		//call the API and decode the response
		$url = "https://www.behance.net/v2/users/" . $this->user_id . "/projects?api_key=" . $this->api_key . "&per_page=" . $count;

		$transient_name = 'essgrid_' . md5($url);
		if ($this->transient_sec > 0 && false !== ($data = get_transient($transient_name)))
			return ($data);

		$rsp = json_decode(wp_remote_fopen($url));
		if (!empty($rsp)) {
			$this->behance_output_array($rsp);
			set_transient($transient_name, $this->stream, $this->transient_sec);
			return $this->stream;
		}
	}

	/**
	 * Get Projects from Channel as Options for Selectbox
	 *
	 * @since    3.0
	 */
	public function get_dribbble_projects_options($current_project = "")
	{
		//call the API and decode the response
		$url = 'https://api.dribbble.com/v1/users/' . $this->user_id . '/projects?access_token=' . $this->client_access_token;
		$rsp = json_decode(wp_remote_fopen($url));

		$return = array();
		if (is_array($rsp))
			foreach ($rsp as $project) {
				$return[] = '<option ' . selected($project->id, $current_project, false) . ' value="' . $project->id . '">' . $project->name . '</option>"';
			}
		else
			$return = "";

		return $return;
	}

	/**
	 * Get Buckets from Channel as Options for Selectbox
	 *
	 * @since    3.0
	 */
	public function get_dribbble_buckets_options($current_project = "")
	{
		//call the API and decode the response
		$url = 'https://api.dribbble.com/v1/users/' . $this->user_id . '/buckets?access_token=' . $this->client_access_token;
		$rsp = json_decode(wp_remote_fopen($url));

		$return = array();
		if (is_array($rsp))
			foreach ($rsp as $bucket) {
				$return[] = '<option ' . selected($bucket->id, $current_bucket, false) . ' value="' . $bucket->id . '">' . $bucket->name . '</option>"';
			}
		else
			$return = "";

		return $return;
	}

	/**
	 * Get Images from single Project
	 *
	 * @since    3.0
	 */
	public function get_dribbble_project_images($project = "", $count = 100)
	{
		//call the API and decode the response
		if (!empty($project)) {
			$url = "https://www.behance.net/v2/projects/" . $project . "?api_key=" . $this->api_key . "&per_page=" . $count;

			$transient_name = 'essgrid_' . md5($url);
			if ($this->transient_sec > 0 && false !== ($data = get_transient($transient_name)))
				return ($data);

			$rsp = json_decode(wp_remote_fopen($url));
			if (!empty($rsp)) {
				$this->behance_images_output_array($rsp, $count);
				set_transient($transient_name, $this->stream, $this->transient_sec);
				return $this->stream;
			}
		}
	}

	/**
	 * Prepare output array $stream for Behance images
	 *
	 * @param string $videos Behance Output Data
	 * @since    3.0
	 */
	private function dribbble_images_output_array($images, $count)
	{
		if (is_object($images)) {
			foreach ($images->project->modules as $image) {
				if (!$count--) break;
				$stream = array();

				if ($image->type == "image") {
					$image_url = @array(
						'disp' => array($image->sizes->disp),
						'max_86400' => array($image->sizes->max_86400),
						'max_1240' => array($image->sizes->max_1240),
						'original' => array($image->sizes->original),
					);

					$stream['custom-image-url'] = $image_url;
					$stream['custom-type'] = 'image'; //image, vimeo, youtube, soundcloud, html
					$stream['post-link'] = $images->project->url;
					$stream['title'] = $images->project->name;
					$stream['content'] = $images->project->name;
					$stream['date'] = date_i18n(get_option('date_format'), strtotime($images->project->modified_on));
					$stream['date_modified'] = date_i18n(get_option('date_format'), strtotime($images->project->modified_on));
					$stream['author_name'] = $images->project->owners[0]->first_name;
					$this->stream[] = $stream;
				}
			}
		}
	}

	/**
	 * Prepare output array $stream for Behance Projects
	 *
	 * @param string $videos Behance Output Data
	 * @since    3.0
	 */
	private function dribbble_output_array($images)
	{
		if (is_object($images)) {
			foreach ($images->projects as $image) {
				$stream = array();

				$image_url = @array(
					'115' => array($image->covers->{'115'}),
					'202' => array($image->covers->{'202'}),
					'230' => array($image->covers->{'230'}),
					'404' => array($image->covers->{'404'}),
					'original' => array($image->covers->original),
				);
				$stream['custom-image-url'] = $image_url;

				$stream['custom-type'] = 'image'; //image, vimeo, youtube, soundcloud, html
				$stream['post-link'] = $image->url;
				$stream['title'] = $image->name;
				$stream['content'] = $image->name;
				$stream['date'] = $image->modified_on;
				$stream['date_modified'] = $image->modified_on;
				$stream['author_name'] = 'dude';
				$this->stream[] = $stream;
			}
		}
	}
}
